<?php
/*
Plugin Name: Get Shipping Zone ID Method ID & Zone Name
Description: This plugin enables you to display shipping methods on the single product page using WooCommerce.
Plugin URI: https://wpsites.net/?p=121707
Version: 1.0.4
Author: Brad Dalton WP SITES
Author URI: https://wpsites.net/bradley-james-dalton-wordpress-developer/
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; 
}

// Enqueue AJAX script for shipping method updates
add_action( 'wp_enqueue_scripts', 'enqueue_shipping_method_ajax_script' );
function enqueue_shipping_method_ajax_script() {
    // Enqueue jQuery if not already included
    wp_enqueue_script( 'jquery' );

    // Enqueue custom script
    wp_enqueue_script( 'shipping-method-ajax', plugin_dir_url(__FILE__) . 'ajax.js', array( 'jquery' ), null, true );

    // Localize script for AJAX URL
    wp_localize_script( 'shipping-method-ajax', 'shippingMethodAjax', array(
        'ajax_url' => admin_url( 'admin-ajax.php' )
    ));
}

// Handle AJAX request to update shipping method
add_action('wp_ajax_update_shipping_zone', 'update_shipping_zone');
add_action('wp_ajax_nopriv_update_shipping_zone', 'update_shipping_zone');

function update_shipping_zone() {
    if (!isset($_POST['selected_method'])) {
        wp_send_json_error('No shipping method selected.');
    }

    $selected_method = sanitize_text_field($_POST['selected_method']);
    
    // Store the chosen shipping method in the session
    WC()->session->set('chosen_shipping_methods', [$selected_method]);
    WC()->cart->calculate_totals(); // Ensure cart updates correctly

    // Get the shipping zone ID associated with the method
    $zone_id = get_shipping_zone_by_method_id($selected_method);
    $zone_name = $zone_id ? get_shipping_zone_name_by_id($zone_id) : 'Unknown';

    // Build the response message
    $response_message = "Selected shipping method ID: $selected_method Zone ID: $zone_id - $zone_name";

    // Return the correct response
    wp_send_json_success($response_message);
}



function get_shipping_zone_by_method_id($shipping_method_id) {
    // Get all shipping zones
    $zones = WC_Shipping_Zones::get_zones();

    // Loop through each zone to find the shipping method
    foreach ($zones as $zone_data) {
        $zone = new WC_Shipping_Zone($zone_data['id']);
        $shipping_methods = $zone->get_shipping_methods(); // Get shipping methods for this zone

        foreach ($shipping_methods as $method) {
            // Construct the full method ID (e.g., 'flat_rate:5')
            $full_method_id = $method->id . ':' . $method->instance_id;

            if ($full_method_id === $shipping_method_id) {
                return $zone->get_id(); // Return the zone ID
            }
        }
    }

    // Also check the "Rest of the World" zone (zone ID 0)
    $default_zone = new WC_Shipping_Zone(0);
    $default_methods = $default_zone->get_shipping_methods();

    foreach ($default_methods as $method) {
        $full_method_id = $method->id . ':' . $method->instance_id;
        if ($full_method_id === $shipping_method_id) {
            return $default_zone->get_id();
        }
    }

    return null; // Return null if the shipping method is not found in any zone
}


// Get zone name by ID
function get_shipping_zone_name_by_id($zone_id) {

    if (!is_numeric($zone_id) || $zone_id < 0) {
        return null;
    }

    $shipping_zone = new WC_Shipping_Zone($zone_id);
    return $shipping_zone->get_zone_name();
}

// Display shipping zone info on cart page for admins
add_action('woocommerce_before_cart', 'display_shipping_zone_for_selected_method');
function display_shipping_zone_for_selected_method() {

    if (is_admin() || !is_user_logged_in() || !current_user_can('manage_woocommerce')) {
        return;
    }

    $chosen_methods = WC()->session->get('chosen_shipping_methods');
    $selected_method = is_array($chosen_methods) ? reset($chosen_methods) : null;

    if ($selected_method) {
        $zone_id = get_shipping_zone_by_method_id($selected_method);
        $zone_name = get_shipping_zone_name_by_id($zone_id);

        if ($zone_id !== null) {
            echo '<pre style="background:#ffebcd;padding:10px;border:1px solid #ffcc00;">';
            echo '<div id="shipping-zone-info">';
            echo 'Selected shipping method ID: <strong>' . esc_html($selected_method) . '</strong>';
            echo ' Zone ID: <strong>' . esc_html($zone_id) . '</strong> - ' . esc_html($zone_name);
            echo '</div></pre>';
        } else {
            echo 'Shipping method not found in any zone.';
        }
    } else {
        echo 'No shipping method selected.';
    }
}
