<?php
/**
 * Plugin Name: Conditional Free Shipping Packets & Boxes V2
 * Plugin URI: https://wp.me/pg7juh-xGZ
 * Description: Custom plugin for conditional shipping prices including free shipping
 * Author: Brad Dalton - WP SITES
 * Author URI: https://wpsites.net/bradley-james-dalton-wordpress-developer/
 * Version: 2
 * Requires Plugins: woocommerce
 */

// Debug: Test if plugin is loading
add_action('init', function() {
    error_log('Custom Shipping Plugin Loaded Successfully');
});

// Add admin menu under WooCommerce
add_action('admin_menu', 'custom_shipping_admin_menu');
function custom_shipping_admin_menu() {
    add_submenu_page(
        'woocommerce',
        'Custom Shipping Labels',
        'Shipping Labels',
        'manage_woocommerce',
        'custom-shipping-labels',
        'custom_shipping_admin_page'
    );
}

// Register settings
add_action('admin_init', 'custom_shipping_settings_init');
function custom_shipping_settings_init() {
    register_setting('custom_shipping_group', 'custom_shipping_labels');
    
    add_settings_section(
        'custom_shipping_section',
        'Shipping Method Labels',
        'custom_shipping_section_callback',
        'custom-shipping-labels'
    );
    
    add_settings_field(
        'free_shipping_100_label',
        'Free Shipping (100+ items) Label',
        'custom_shipping_field_callback',
        'custom-shipping-labels',
        'custom_shipping_section',
        array('field' => 'free_shipping_100_label')
    );
    
    add_settings_field(
        'box_shipping_label',
        'Boxes Shipping Label',
        'custom_shipping_field_callback',
        'custom-shipping-labels',
        'custom_shipping_section',
        array('field' => 'box_shipping_label')
    );
    
    add_settings_field(
        'packet_shipping_label',
        'Packets Shipping (< 5) Label',
        'custom_shipping_field_callback',
        'custom-shipping-labels',
        'custom_shipping_section',
        array('field' => 'packet_shipping_label')
    );
    
    add_settings_field(
        'free_packet_shipping_label',
        'Free Shipping (5+ Packets) Label',
        'custom_shipping_field_callback',
        'custom-shipping-labels',
        'custom_shipping_section',
        array('field' => 'free_packet_shipping_label')
    );
}

// Admin page callback
function custom_shipping_admin_page() {
    ?>
    <div class="wrap">
        <h1>Custom Shipping Labels</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('custom_shipping_group');
            do_settings_sections('custom-shipping-labels');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// Section callback
function custom_shipping_section_callback() {
    echo '<p>Customize the labels for your shipping methods. Leave blank to use default labels.</p>';
}

// Field callback
function custom_shipping_field_callback($args) {
    $options = get_option('custom_shipping_labels', array());
    $field = $args['field'];
    $value = isset($options[$field]) ? $options[$field] : '';
    
    $default_labels = array(
        'free_shipping_100_label' => 'Free Shipping (100+ items)',
        'box_shipping_label' => 'Shipping (Boxes)',
        'packet_shipping_label' => 'Shipping (Packets < 5)',
        'free_packet_shipping_label' => 'Free Shipping (5+ Packets)'
    );
    
    echo '<input type="text" name="custom_shipping_labels[' . $field . ']" value="' . esc_attr($value) . '" style="width: 300px;" />';
    echo '<p class="description">Default: ' . $default_labels[$field] . '</p>';
}

// Get custom label or default
function get_custom_shipping_label($key, $default) {
    $options = get_option('custom_shipping_labels', array());
    return isset($options[$key]) && !empty($options[$key]) ? $options[$key] : $default;
}

// Use woocommerce_package_rates to filter and show only applicable shipping methods
add_filter('woocommerce_package_rates', 'custom_shipping_rules_based_on_category_and_quantity', 10, 2);
function custom_shipping_rules_based_on_category_and_quantity($rates, $package) {
    
    // Debug: Log when function is called
    error_log('Custom shipping function called');
    
    // Safety check - ensure we have valid data
    if (empty($package) || !isset($package['contents']) || !is_array($package['contents'])) {
        error_log('Invalid package data in custom shipping function');
        return $rates; // Return original rates if invalid data
    }

    $packets_category = 'packets'; // category slug
    $boxes_category   = 'boxes';   // category slug

    $packet_qty = 0;
    $box_qty = 0;
    $total_qty = 0;

    // Calculate quantities for each category and total
    foreach ($package['contents'] as $item) {
        if (!isset($item['data']) || !isset($item['quantity'])) {
            continue; // Skip invalid items
        }
        
        $product = $item['data'];
        $qty = $item['quantity'];
        $total_qty += $qty;

        // Get product categories safely
        try {
            $terms = get_the_terms($product->get_id(), 'product_cat');
            if (empty($terms) || is_wp_error($terms)) continue;

            foreach ($terms as $term) {
                if ($term->slug === $packets_category) {
                    $packet_qty += $qty;
                } elseif ($term->slug === $boxes_category) {
                    $box_qty += $qty;
                }
            }
        } catch (Exception $e) {
            error_log('Error getting product terms: ' . $e->getMessage());
            continue;
        }
    }

    // Debug: Log quantities
    error_log("Quantities - Total: $total_qty, Packets: $packet_qty, Boxes: $box_qty");

    // If no rates exist, return empty array
    if (empty($rates)) {
        error_log('No shipping rates available');
        return $rates;
    }

    // Get the first available rate to modify
    $first_rate = reset($rates);
    if (!$first_rate) {
        error_log('No valid shipping rate found');
        return $rates;
    }

    $new_rates = array();

    // Rule 3: Free shipping if total cart quantity >= 100 (highest priority)
    if ($total_qty >= 100) {
        // Create a free shipping rate
        $free_rate = clone $first_rate;
        $free_rate->set_cost(0);
        $free_rate->set_label(get_custom_shipping_label('free_shipping_100_label', 'Free Shipping (100+ items)'));
        $new_rates['free_shipping_100'] = $free_rate;
        error_log('Applied free shipping for 100+ items');
        return $new_rates;
    }

    // Rule 2: If any boxes in cart, apply $8 shipping (second priority)
    if ($box_qty > 0) {
        // Create a box shipping rate
        $box_rate = clone $first_rate;
        $box_rate->set_cost(8.00);
        $box_rate->set_label(get_custom_shipping_label('box_shipping_label', 'Shipping (Boxes)'));
        $new_rates['box_shipping'] = $box_rate;
        error_log('Applied $8 shipping for boxes');
        return $new_rates;
    }

    // Rule 1: Only packets in cart (lowest priority)
    if ($packet_qty > 0) {
        if ($packet_qty < 5) {
            // Create a packet shipping rate
            $packet_rate = clone $first_rate;
            $packet_rate->set_cost(4.00);
            $packet_rate->set_label(get_custom_shipping_label('packet_shipping_label', 'Shipping (Packets < 5)'));
            $new_rates['packet_shipping'] = $packet_rate;
            error_log('Applied $4 shipping for packets < 5');
        } else {
            // Create a free packet shipping rate
            $free_packet_rate = clone $first_rate;
            $free_packet_rate->set_cost(0);
            $free_packet_rate->set_label(get_custom_shipping_label('free_packet_shipping_label', 'Free Shipping (5+ Packets)'));
            $new_rates['free_packet_shipping'] = $free_packet_rate;
            error_log('Applied free shipping for 5+ packets');
        }
        return $new_rates;
    }

    // Default fallback — return original rates if no custom rules apply
    error_log('No custom shipping rules applied, using default rates');
    return $rates;
}
